﻿using Microsoft.EntityFrameworkCore;
using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LinqToSqlite
{
    class Osoba : ICloneable //DTO, POCO
    {
        [Key] public int Id { get; set; } //primary key
        public string Imię { get; set; }
        public string Nazwisko { get; set; }
        public int NumerTelefonu { get; set; }
        public int Wiek { get; set; }

        public Adres Adres { get; set; } //relacja z bazą danych

        public override string ToString()
        {
            return $"{Id}. {Imię} {Nazwisko} ({Wiek}), nr telefonu: {NumerTelefonu}, adres: {Adres}";
        }

        public object Clone()
        {
            return new Osoba() { Imię = this.Imię, Nazwisko = this.Nazwisko, Wiek = this.Wiek, NumerTelefonu = this.NumerTelefonu };
        }

        public static readonly Osoba OsobaPusta = new Osoba() { Id = -1, Imię = "", Nazwisko = "", NumerTelefonu = -1, Wiek = -1 };

        public override bool Equals(object? obj)
        {
            if (!(obj is Osoba)) return false;
            Osoba innaOsoba = (Osoba)obj;
            return
                innaOsoba.Id == Id &&
                innaOsoba.Imię.Equals(Imię) &&
                innaOsoba.Nazwisko.Equals(Nazwisko);
        }

        public override int GetHashCode()
        {
            return Id.GetHashCode() ^ Imię.GetHashCode() ^ Nazwisko.GetHashCode();
        }
    }

    class Adres
    {
        [Key] public int Id {get; set;}
        public string Miasto { get; set; }
        public string Ulica { get; set; }
        public int NumerDomu { get; set; }
        public int? NumerMieszkania { get; set; }

        public override string ToString()
        {
            return $"{Id}. {Miasto}, {Ulica} {NumerDomu}" + ((NumerMieszkania.HasValue) ? $"/{NumerMieszkania}" : "");
        }

        public override bool Equals(object? obj)
        {
            if (!(obj is Adres)) return false;
            Adres innyAdres = (Adres)obj;
            return
                innyAdres.Miasto.Equals(Miasto) &&
                innyAdres.Ulica.Equals(Ulica) &&
                innyAdres.NumerDomu.Equals(NumerDomu) &&
                innyAdres.NumerMieszkania.Equals(NumerMieszkania);
        }

        public override int GetHashCode()
        {
            return Miasto.GetHashCode() ^ Ulica.GetHashCode() ^ NumerDomu.GetHashCode() ^ NumerMieszkania.GetHashCode();
        }
    }

    static class Rozszerzenia
    {
        public static void Wyświetl<T>(this IEnumerable<T> list)
        {
            Console.WriteLine("Lista obiektów: ");
            foreach (T element in list)
            {
                if (element != null) Console.WriteLine(element.ToString());
                else Console.WriteLine("---");
            }
            Console.WriteLine();
        }
    }

    class BazaDanychOsóbDbContext : DbContext
    {
        public DbSet<Osoba> Osoby { get; set; }
        public DbSet<Adres> Adresy { get; set; }

        protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
        {
            //base.OnConfiguring(optionsBuilder);
            optionsBuilder.UseSqlite("Data Source=osoby.db");
        }
    }

    class BazaDanychOsób : IDisposable
    {
        private BazaDanychOsóbDbContext dbc = new BazaDanychOsóbDbContext();

        public BazaDanychOsób()
        {
            dbc.Database.EnsureCreated();
        }

        public void Dispose()
        {
            dbc.Dispose();
        }

#if DEBUG
        public Osoba[] Osoby { get => dbc.Osoby.Include(o => o.Adres).ToArray(); }
        public Adres[] Adresy { get => dbc.Adresy.ToArray(); }
#endif

        #region CRUD = Create+, Read+, Update-, Delete+
        public Osoba PobierzOsobę(int idOsoby)
        {            
            return dbc.Osoby.Include(o => o.Adres).FirstOrDefault(o => o.Id == idOsoby);
        }

        //indekser
        public Osoba this[int idOsoby]
        {
            get => PobierzOsobę(idOsoby);
            set => ZmieńDaneOsoby(idOsoby, value);
        }

        public int[] IdentyfikatoryOsób
        {
            get => dbc.Osoby/*.OrderBy(o => o.Id)*/.Select(o => o.Id).OrderBy(id => id).ToArray();            
        }

        public int DodajOsobę(Osoba osoba) //identyfikator w argumencie będzie ignorowany
        {
            if(osoba == null)
                throw new ArgumentNullException("Podano pusty obiekt");
            if (osoba.Adres == null)
                throw new ArgumentNullException("Osoba musi mieć adres zamieszkania");
            if (dbc.Osoby.Any(o => o.Equals(osoba)))
            {
                //throw new ArgumentException("W bazie istnieje osoba o tym samym identyfikatorze i personaliach");
                return osoba.Id;
            }
            /*
            else
            {
                //ustalamy id
                int newId = 0; //TODO: sprawdzić, czy baza nadaje dobre id
                if(dbc.Osoby.Count() > 0) newId = IdentyfikatoryOsób.Max() + 1;
                osoba.Id = newId;
            }
            */

            //unikanie dublowania adresów
            Adres adres = dbc.Adresy.ToArray().FirstOrDefault(a => a.Equals(osoba.Adres));
            if (adres != null) osoba.Adres = adres;

            dbc.Osoby.Add(osoba);
            dbc.SaveChanges();

            return osoba.Id;
        }

        private void usuńNieużywaneAdresy()
        {
            int[] identyfikatoryUżywanychAdresów = dbc.Osoby.Select(o => o.Adres.Id).Distinct().ToArray();
            List<Adres> nieużywaneAdresy = new List<Adres>();
            foreach(Adres adres in dbc.Adresy)
            {
                if(!identyfikatoryUżywanychAdresów.Contains(adres.Id))
                    nieużywaneAdresy.Add(adres);
            }

            dbc.Adresy.RemoveRange(nieużywaneAdresy);
            dbc.SaveChanges();
        }

        public void UsuńOsobę(int idOsoby)
        {
            Osoba osoba = PobierzOsobę(idOsoby);
            if(osoba != null)
            {
                dbc.Osoby.Remove(osoba);
                dbc.SaveChanges();
                usuńNieużywaneAdresy();
            }
        }

        public void ZmieńDaneOsoby(int idOsoby, Osoba noweDaneOsoby)
        {
            Osoba osoba = PobierzOsobę(idOsoby);
            if (osoba == null) throw new Exception("Niepoprawny identyfikator");
            if (noweDaneOsoby == null) throw new ArgumentNullException("Nowe dane nie mogą być puste");
            osoba.Imię = noweDaneOsoby.Imię;
            osoba.Nazwisko = noweDaneOsoby.Nazwisko;
            osoba.Wiek = noweDaneOsoby.Wiek;
            osoba.NumerTelefonu = noweDaneOsoby.NumerTelefonu;
            /*
            osoba.Adres.Miasto = noweDaneOsoby.Adres.Miasto;
            osoba.Adres.Ulica = noweDaneOsoby.Adres.Ulica;
            osoba.Adres.NumerDomu = noweDaneOsoby.Adres.NumerDomu;
            osoba.Adres.NumerMieszkania = noweDaneOsoby.Adres.NumerMieszkania;
            */            
            dbc.SaveChanges();
        }
        #endregion
    }
}
