﻿using System;
using System.IO;
using System.Text;
using Microsoft.WindowsAzure.Storage;
using Microsoft.WindowsAzure.Storage.Blob;
using SharedStorage;

namespace StorageExample
{
    internal class Program
    {
        private static void Main(string[] args)
        {
            var account = CloudStorageAccount.Parse(StorageConst.StorageConnectionString);
            var client = account.CreateCloudBlobClient();
            var container = client.GetContainerReference(StorageConst.ContainerName);

            container.CreateIfNotExists();
            Console.WriteLine(TokenForContainer(container));

            UploadText(container);
            UploadByte(container);
            UploadFileStream(container);

            ListContainer(container);
            DownloadFromContainer(container);

            var blobToken = TokenForContainer(container);
            CloudBlobContainer containerWithSas = new CloudBlobContainer(new Uri(blobToken));
            String message = "It should throw error";
            CloudBlockBlob blob = containerWithSas.GetBlockBlobReference("test.txt");
            blob.UploadText(message);

            Console.ReadKey();


        }

        private static string TokenForContainer(CloudBlobContainer container)
        {
            var sasConstraints = new SharedAccessBlobPolicy();
            sasConstraints.SharedAccessExpiryTime = DateTime.UtcNow.AddHours(4);
            sasConstraints.Permissions = SharedAccessBlobPermissions.Read | SharedAccessBlobPermissions.Write;

            //Generate the shared access signature on the container, setting the constraints directly on the signature.
            var sasContainerToken = container.GetSharedAccessSignature(sasConstraints);

            //Return the URI string for the container, including the SAS token.
            return container.Uri + sasContainerToken;
        }

        private static void DownloadFromContainer(CloudBlobContainer container)
        {
        }

        private static void ListContainer(CloudBlobContainer container)
        {
            var blobItems = container.ListBlobs();
            foreach (var blobItem in blobItems)
            {
                switch (blobItem.GetType().Name)
                {
                    case "CloudBlockBlob":
                        var block = blobItem as CloudBlockBlob;
                        block.DownloadToFile(block.Name, FileMode.Create);
                        var sasConstraints = new SharedAccessBlobPolicy();
                        sasConstraints.SharedAccessExpiryTime = DateTime.UtcNow.AddHours(4);
                        sasConstraints.Permissions = SharedAccessBlobPermissions.Read |
                                                     SharedAccessBlobPermissions.Write;
                        break;
                    case "CloudPageBlob":
                        var pageBlob = blobItem as CloudPageBlob;
                        pageBlob.DownloadToFile(pageBlob.Name, FileMode.Create);
                        break;
                }
            }
        }

        private static void UploadFileStream(CloudBlobContainer container)
        {
            var fileName = Guid.NewGuid().ToString("N") + Path.GetExtension("ElasticScale.zip");
            var blob = container.GetBlockBlobReference(fileName);
            using (var stream = new FileStream("ElasticScale.zip", FileMode.Open))
            {
                blob.UploadFromStream(stream);
            }
        }

        private static void UploadByte(CloudBlobContainer container)
        {
            var text = "Hello world from Azure Storage";
            var byteArray = Encoding.UTF8.GetBytes(text);
            var blob = container.GetBlockBlobReference("bytes.txt");
            blob.UploadFromByteArray(byteArray, 0, byteArray.Length);
        }

        private static void UploadText(CloudBlobContainer container)
        {
            var text = "Hello world from Azure Storage";
            var blob = container.GetBlockBlobReference("text.txt");
            blob.UploadText(text, Encoding.UTF8);
        }
    }
}